-- script contributed by Fabian Wenzel
-- Bachelor thesis Jan 2018

-----------------------------------------------
--Calculation of NiO RIXS conductivity tensor--
--L3 edge in Octahedral symmetry Oh			 --
-----------------------------------------------

Verbosity(0)
-------------------------------------------------------------------------------------------------------------------------------------
--define function that takes a matrix and returns an operator acting on indices from startIndex to startIndex + dimension of matrix--
-------------------------------------------------------------------------------------------------------------------------------------

function MatrixToOperator(matrix, startIndex)
  operator = 0
  for i = 1, #matrix do
    for j = 1, #matrix do
      weight = matrix[i][j]
      operator = operator + NewOperator("Number", #matrix+startIndex, i + startIndex - 1, j + startIndex - 1)*weight
    end
  end
  operator.Chop()
  return operator
end

--define function that copies matrices by value not by reference

function CloneMatrix(matrix)
    out = {}
    for i = 1, #matrix do
        out[i] = {}
        for j = 1, #matrix[i] do
            out[i][j] = matrix[i][j]
        end
    end
    return out
end

--define function that restricts a matrix to the given spin orbitals indices (starting from 0 instead of 1 for array indices). All other rows in matrix are set to zero.

function RestrictMatrix(matrix, indices)
    out = CloneMatrix(matrix)  
    for i = 0, #out - 1 do
        bool = false
        for index = 1, #indices do
            if indices[index] == i then
                bool = true
            end
        end
        if not bool then
            for j = 0, #out[i] - 1 do
                out[i + 1][j + 1] = 0
            end
        end
    end
    return out
end

--define function that combines two matrices into block diagonal matrix

function CombineBlockMatrices(matrix1, matrix2)
    out = {}
    for i = 1, #matrix1 + #matrix2 do
        out[i] = {}
        for j =1, #matrix1[1] + #matrix2[1] do
        --insert block for 2p rotation 
            if i < #matrix1 + 1 and j < #matrix1[1] + 1 then
            out[i][j] = matrix1[i][j]
        --insert block for 3d rotation
            elseif i >= #matrix1 + 1 and j >= #matrix1[1] + 1 then
            out[i][j] = matrix2[i-#matrix1][j-#matrix1[1]]
        --any other element is set to zero    
            else
            out[i][j] = 0
            end
        end
    end
    return out
end

----------------------------------------------------
--standard definitions like in other NiO tutorials--
----------------------------------------------------

--define our Hilbert space with 6 2p and 10 3d orbitals

NF=16
NB=0
IndexDn_2p={0,2,4}
IndexUp_2p={1,3,5}
IndexDn_3d={6,8,10,12,14}
IndexUp_3d={7,9,11,13,15}

-- just like in the previous examples we define several operators acting on the Ni -3d shell

OppSx   =NewOperator("Sx"   ,NF, IndexUp_3d, IndexDn_3d)
OppSy   =NewOperator("Sy"   ,NF, IndexUp_3d, IndexDn_3d)
OppSz   =NewOperator("Sz"   ,NF, IndexUp_3d, IndexDn_3d)
OppSsqr =NewOperator("Ssqr" ,NF, IndexUp_3d, IndexDn_3d)
OppSplus=NewOperator("Splus",NF, IndexUp_3d, IndexDn_3d)
OppSmin =NewOperator("Smin" ,NF, IndexUp_3d, IndexDn_3d)

OppLx   =NewOperator("Lx"   ,NF, IndexUp_3d, IndexDn_3d)
OppLy   =NewOperator("Ly"   ,NF, IndexUp_3d, IndexDn_3d)
OppLz   =NewOperator("Lz"   ,NF, IndexUp_3d, IndexDn_3d)
OppLsqr =NewOperator("Lsqr" ,NF, IndexUp_3d, IndexDn_3d)
OppLplus=NewOperator("Lplus",NF, IndexUp_3d, IndexDn_3d)
OppLmin =NewOperator("Lmin" ,NF, IndexUp_3d, IndexDn_3d)

OppJx   =NewOperator("Jx"   ,NF, IndexUp_3d, IndexDn_3d)
OppJy   =NewOperator("Jy"   ,NF, IndexUp_3d, IndexDn_3d)
OppJz   =NewOperator("Jz"   ,NF, IndexUp_3d, IndexDn_3d)
OppJsqr =NewOperator("Jsqr" ,NF, IndexUp_3d, IndexDn_3d)
OppJplus=NewOperator("Jplus",NF, IndexUp_3d, IndexDn_3d)
OppJmin =NewOperator("Jmin" ,NF, IndexUp_3d, IndexDn_3d)

Oppldots=NewOperator("ldots",NF, IndexUp_3d, IndexDn_3d)

OppF0 =NewOperator("U", NF, IndexUp_3d, IndexDn_3d, {1,0,0})
OppF2 =NewOperator("U", NF, IndexUp_3d, IndexDn_3d, {0,1,0})
OppF4 =NewOperator("U", NF, IndexUp_3d, IndexDn_3d, {0,0,1})

Oppcldots= NewOperator("ldots", NF, IndexUp_2p, IndexDn_2p)

OppUpdF0 = NewOperator("U", NF, IndexUp_2p, IndexDn_2p, IndexUp_3d, IndexDn_3d, {1,0}, {0,0})
OppUpdF2 = NewOperator("U", NF, IndexUp_2p, IndexDn_2p, IndexUp_3d, IndexDn_3d, {0,1}, {0,0})
OppUpdG1 = NewOperator("U", NF, IndexUp_2p, IndexDn_2p, IndexUp_3d, IndexDn_3d, {0,0}, {1,0})
OppUpdG3 = NewOperator("U", NF, IndexUp_2p, IndexDn_2p, IndexUp_3d, IndexDn_3d, {0,0}, {0,1})

Akm = {{1,-1,sqrt(1/2)},{1, 1,-sqrt(1/2)}}
TXASx_spherical = NewOperator("CF", NF, IndexUp_3d, IndexDn_3d, IndexUp_2p, IndexDn_2p, Akm)

Akm = {{1,-1,sqrt(1/2)*I},{1, 1,sqrt(1/2)*I}}
TXASy_spherical = NewOperator("CF", NF, IndexUp_3d, IndexDn_3d, IndexUp_2p, IndexDn_2p, Akm)

Akm = {{1,0,1}}
TXASz_spherical = NewOperator("CF", NF, IndexUp_3d, IndexDn_3d, IndexUp_2p, IndexDn_2p, Akm)

U       =  0.000 
F2dd    = 11.142 
F4dd    =  6.874
F0dd    = U+(F2dd+F4dd)*2/63
Upd     =  0.000
F2pd    =  6.667
G1pd    =  4.922
G3pd    =  2.796
F0pd    =  Upd + G1pd*1/15 + G3pd*3/70


----------------------------------------------------------------------------------------------------------------------
--next, we need to define rotation matrices from spherical harmonics to cubic harmonics keeping spin order untouched--
----------------------------------------------------------------------------------------------------------------------

--for p orbitals we convert {P-, P0, P+} to {j12-, j12+, j32--, j32-, j32+, j32++}

d = math.sqrt(1/3)
e = math.sqrt(2/3)

rotMat2p = {{0,-e, d, 0, 0, 0} , 
			{0, 0, 0,-d, e, 0} , 
			{1, 0, 0, 0, 0, 0} , 
			{0, d, e, 0, 0, 0} , 
			{0, 0, 0, e, d, 0} , 
			{0, 0, 0, 0, 0, 1}
}
	
--for d orbitals we convert {P-2, P-1, P0, P+1, P+2} to {Dzz, Dxx-yy, Dxz, Dyz, Dxy}

c = 1/math.sqrt(2)
I = Complex.I()
a = c*I
		
rotMat3d = {{0	, 0	, 0	, 0 , 1	, 0	, 0	, 0	, 0	, 0 },
			{0	, 0	, 0	, 0	, 0	, 1	, 0	, 0	, 0	, 0	},
			{c	, 0	, 0	, 0	, 0	, 0	, 0	, 0	, c , 0	},
			{0	, c	, 0	, 0	, 0	, 0	, 0	, 0	, 0	, c	},
			{0	, 0	, c	, 0	, 0	, 0	,-c	, 0	, 0	, 0	},
			{0	, 0	, 0	, c , 0	, 0	, 0	,-c , 0	, 0	},
			{0	, 0	,-a	, 0	, 0	, 0	,-a	, 0	, 0	, 0	},
			{0	, 0	, 0	,-a , 0	, 0	, 0	,-a , 0	, 0	},
			{-a	, 0	, 0	, 0	, 0	, 0	, 0	, 0	, a , 0	},
			{0	, -a, 0	, 0	, 0	, 0	, 0	, 0	, 0	, a	}
}

--now for concluding to full rotation matrix we need to combine those two blocks to a 16x16 rotation matrix

rotMat = CombineBlockMatrices(rotMat2p, rotMat3d)

--now we can rotate the basis of some operators

OppSz_cubic = Rotate(OppSz, rotMat)
OppSsqr_cubic = Rotate(OppSsqr, rotMat)
OppLz_cubic = Rotate(OppLz, rotMat)
OppLsqr_cubic = Rotate(OppLsqr, rotMat)
OppJsqr_cubic = Rotate(OppJsqr, rotMat)
Oppldots_cubic = Rotate(Oppldots, rotMat)
OppF2_cubic = Rotate(OppF2, rotMat)
OppF4_cubic = Rotate(OppF4, rotMat)

--and create operators for counting eg and t2g electrons (they are diagonal in cubic basis) 

OppNeg_cubic = 0
for i = 6, 9 do
  OppNeg_cubic = OppNeg_cubic + NewOperator("Cr", NF, i)*NewOperator("An", NF, i)
end


OppNt2g_cubic = 0
for i = 10, 15 do
  OppNt2g_cubic = OppNt2g_cubic + NewOperator("Cr", NF, i)*NewOperator("An", NF, i)
end

----------------------------------------------------------
--define our Hamiltonians and calculate the ground state--
----------------------------------------------------------

tenDq   =  1.1
zeta_3d = 0*0.081
zeta_2p = 11.498
Bz      = 0*0.000001

-- the ground state Hamiltonian is the sum of the different operators multiplied with their prefactor
Hamiltonian = F0dd*OppF0 + F2dd*OppF2 + F4dd*OppF4 + zeta_3d*Oppldots + Bz*(2*OppSz+OppLz)

--the core hole Hamiltonian has the interactions of the p core hole included
XASHamiltonian = Hamiltonian + zeta_2p * Oppcldots + F0pd * OppUpdF0 + F2pd * OppUpdF2 + G1pd * OppUpdG1 + G3pd * OppUpdG3

--now we can roatate both the whole Hamiltonians into cubic basis
Hamiltonian_cubic = Rotate(Hamiltonian, rotMat)
XASHamiltonian_cubic = Rotate(XASHamiltonian, rotMat)

--and add the Crystal Field operator represented in the basis of cubic harmonics (here it becomes diagonal)

Eeg = 0.6*tenDq
Et2g = -0.4*tenDq

MattenDq = {{Eeg	,0	,0	,0	,0	,0	,0	,0	,0	,0	},
			{0	,Eeg	,0	,0	,0	,0	,0	,0	,0	,0	},
			{0	,0	,Eeg	,0	,0	,0	,0	,0	,0	,0	},
			{0	,0	,0	,Eeg	,0	,0	,0	,0	,0	,0	},
			{0	,0	,0	,0	,Et2g	,0	,0	,0	,0	,0	},
			{0	,0	,0	,0	,0	,Et2g	,0	,0	,0	,0	},
			{0	,0	,0	,0	,0	,0	,Et2g	,0	,0	,0	},
			{0	,0	,0	,0	,0	,0	,0	,Et2g	,0	,0	},
			{0	,0	,0	,0	,0	,0	,0	,0	,Et2g	,0	},
			{0	,0	,0	,0	,0	,0	,0	,0	,0	,Et2g	}
}

OpptenDq_cubic = MatrixToOperator(MattenDq, 6)

Hamiltonian_cubic = Hamiltonian_cubic + OpptenDq_cubic
XASHamiltonian_cubic = XASHamiltonian_cubic + OpptenDq_cubic

--now calculate the ground state triplet

Npsi=3
StartRestrictions = {NF, NB, {"111111 0000000000",6,6}, {"000000 1111111111",8,8}}
psiList = Eigensystem(Hamiltonian_cubic, StartRestrictions, Npsi)

--In order to get some information on these eigenstates it is good to plot expectation values
--We first define a list of all the operators we would like to calculate the expectation value of

oppList={	Hamiltonian_cubic, OppSsqr_cubic, OppLsqr_cubic, OppJsqr_cubic, OppSz_cubic, OppLz_cubic, Oppldots_cubic, OppF2_cubic, OppF4_cubic, OppNeg_cubic, OppNt2g_cubic	}; 

-- next we loop over all operators and all states and print the expectation value

print(" <E>    <S^2>  <L^2>  <J^2>  <S_z>  <L_z>  <l.s>  <F[2]> <F[4]>  <Neg>  <Nt2g>");
for i = 1, #psiList do
	for j = 1,#oppList do
	expectationvalue = Chop(psiList[i]*oppList[j]*psiList[i])
	io.write(string.format("%6.3f ",expectationvalue))
	end
	print()
end

-------------------------------------
--calculate the conductivity tensor--
-------------------------------------

TimeStart("Calculation RIXS Tensor")

--define parameters for energy intervals

--absorption close to L3 edge around 4,5eV

Emin1 = -4
Emax1 = -2
NE1 = 2

--RIXS spectrum for d-d excitations interesting from 0eV (elastic scattering) to 7,5eV

Emin2 = -0.5
Emax2 = 7.5
NE2 = 1000

--Here it is possible to restrict the transition operators to certain spin orbitals (here we keep them all)

rotMatT = rotMat --RestrictMatrix(rotMat, {1, 3, 5, 7, 9}) --

--define absorption operators for polarizations x,y,z and rotate them to cubic basis

TXASx = Rotate(TXASx_spherical, rotMatT)
TXASy = Rotate(TXASy_spherical, rotMatT)
TXASz = Rotate(TXASz_spherical, rotMatT)

--emission operators are their conjugate transposed operators

TXASx_dag = Rotate(ConjugateTranspose(TXASx_spherical), rotMatT)
TXASy_dag = Rotate(ConjugateTranspose(TXASy_spherical), rotMatT)
TXASz_dag = Rotate(ConjugateTranspose(TXASz_spherical), rotMatT)


--next, we calculate the RIXS spectra
--to receive the off diagonals we finally need to make linear combinations of diagonal spectra
--for 9 coupled polarizations xx,xy,xz,yx,yy,yz,zx,zy,zz (photon in out), our tensor will contain 81 elements
--these can be calculated out of each 9 linear combinations of the absorption and emission operators T1 and T2

TXASypz  =  sqrt(1/2)*(TXASy + TXASz) 
TXASzpx  =  sqrt(1/2)*(TXASz + TXASx) 
TXASxpy  =  sqrt(1/2)*(TXASx + TXASy) 
TXASypiz =  sqrt(1/2)*(TXASy + I * TXASz) 
TXASzpix =  sqrt(1/2)*(TXASz + I * TXASx) 
TXASxpiy =  sqrt(1/2)*(TXASx + I * TXASy) 

T1 = {TXASx, TXASy, TXASz, TXASypz, TXASzpx, TXASxpy, TXASypiz, TXASzpix, TXASxpiy}

TXASypz_dag  =  sqrt(1/2)*(TXASy_dag + TXASz_dag)
TXASzpx_dag  =  sqrt(1/2)*(TXASz_dag + TXASx_dag)
TXASxpy_dag  =  sqrt(1/2)*(TXASx_dag + TXASy_dag)
TXASypiz_dag =  sqrt(1/2)*(TXASy_dag +I * TXASz_dag)
TXASzpix_dag =  sqrt(1/2)*(TXASz_dag +I * TXASx_dag) 
TXASxpiy_dag =  sqrt(1/2)*(TXASx_dag +I * TXASy_dag)

T2 = {TXASx_dag, TXASy_dag, TXASz_dag, TXASypz_dag, TXASzpx_dag, TXASxpy_dag, TXASypiz_dag, TXASzpix_dag, TXASxpiy_dag}

--all diagonal spectra needed are represented by all 9x9 = 81 combinations of absorption and emission operators in lists T1 and T2
--as we do not want magnetic effects in our tensor, we sum over the ground state triplet for Sz = -1,0,+1 

RIXSSpectra_Smin = CreateResonantSpectra(XASHamiltonian_cubic, Hamiltonian_cubic, T1, T2, psiList[1], {{"Emin1",Emin1}, {"Emax1",Emax1}, {"NE1",NE1}, {"Gamma1",1.0}, {"Emin2",Emin2}, {"Emax2",Emax2}, {"NE2",NE2}, {"Gamma2",0.05}})
RIXSSpectra_S0 = CreateResonantSpectra(XASHamiltonian_cubic, Hamiltonian_cubic, T1, T2, psiList[2], {{"Emin1",Emin1}, {"Emax1",Emax1}, {"NE1",NE1}, {"Gamma1",1.0}, {"Emin2",Emin2}, {"Emax2",Emax2}, {"NE2",NE2}, {"Gamma2",0.05}})
RIXSSpectra_Splus = CreateResonantSpectra(XASHamiltonian_cubic, Hamiltonian_cubic, T1, T2, psiList[3], {{"Emin1",Emin1}, {"Emax1",Emax1}, {"NE1",NE1}, {"Gamma1",1.0}, {"Emin2",Emin2}, {"Emax2",Emax2}, {"NE2",NE2}, {"Gamma2",0.05}})

RIXSSpectra = RIXSSpectra_Smin + RIXSSpectra_S0 + RIXSSpectra_Splus 

--we are not interested in these spectra but in a certain linear combination of them
--we want our tensor of the form e1,e2,e3,e4 with 4 polarizations each composed of x,y,z
--this is done by the following calculation


--this is similar to the calculation of the XAS tensor with only 3x3 = 9 elements
--we use its summation table and to the summation first over the inner operators of the Greens funcion T2, T2_dag then over the outer operators T1, T1_dag
--more information can be found in my Bachelor Thesis, page 31

sumXAS = 	{	{1,0,0,               0,0,0, 0, 0,0}, {-(1-I)/2,-(1-I)/2,0, 0,0,1, 0,0,-I}, {-(1+I)/2,0,-(1+I)/2, 0,1,0, 0,I,0 },
				{-(1+I)/2,-(1+I)/2,0, 0,0,1, 0, 0,I}, {0,1,0, 0,0,0, 0,0,0},                {0,-(1-I)/2,-(1-I)/2, 1,0,0, -I,0,0},
				{-(1-I)/2,0,-(1-I)/2, 0,1,0, 0,-I,0}, {0,-(1+I)/2,-(1+I)/2, 1,0,0, I,0, 0}, {0,0,1, 0,0,0, 0,0,0}					}


--initialize summation tables for summation over out-going polarizations T2	
		 
sumT2 = {}

for t1 = 1, #T1 do
	sumT2[t1] = {}
	for t2 = 1, #T2 do
		sumT2[t1][t2] = {}
		for ne1 = 1, NE1 + 1 do
			sumT2[t1][t2][ne1] = {}
		end
	end
end

for t1 = 1, #T1 do
	for t2 = 1, #T2 do
		for ne1 = 1, NE1 + 1 do
			for i = 1, #sumXAS[t2] do	
				for j = 1, #T1*#T2*(NE1 + 1) do
					if j == (t1 - 1)*#T2*(NE1 + 1) + (i - 1)*(NE1 + 1) + ne1 then
						sumT2[t1][t2][ne1][j] = sumXAS[t2][i]
					elseif sumT2[t1][t2][ne1][j] == nil then
						sumT2[t1][t2][ne1][j] = 0
					end
				end
			end
		end
	end
end

--initialize summation tables for summation over in-coming polarizations T1
	 
sumT1 = {}

for t1 = 1, #T1 do
	sumT1[t1] = {}
	for t2 = 1, #T2 do
		sumT1[t1][t2] = {}
		for ne1 = 1, NE1 + 1 do
			sumT1[t1][t2][ne1] = {}
			for j = 1, #T1*#T2*(NE1 + 1) do
				sumT1[t1][t2][ne1][j] = 0
			end
		end
	end
end

for t2 = 1, #T2 do
	for t1 = 1, #T1 do
		for ne1 = 1, NE1 + 1 do
			for j = 1, #T1*#T2*(NE1 + 1) do
				for i = 1, #sumXAS[t1] do
					sumT1[t1][t2][ne1][j] = sumT1[t1][t2][ne1][j] + sumT2[i][t2][ne1][j] * sumXAS[t1][i]
				end
			end
		end
	end
end

--initialize summation tables in tensor form
--change the order from (in,in_cc|out,out_cc) to (in,out|in_cc,out_cc)

sumTensor = {}

for conjugate = 1, #T1 do
	sumTensor[conjugate] = {}
	for complex = 1, #T2 do
		sumTensor[conjugate][complex] = {}
		for ne1 = 1, NE1 + 1 do
			sumTensor[conjugate][complex][ne1] = {}
		end
	end
end

for element = 0, #T2 - 1 do
	for block = 0, #T1 - 1 do
		for ne1 = 1, NE1 + 1 do
			conjugate = sqrt(#T1)*(block%sqrt(#T1)) + element%sqrt(#T2)
			complex = sqrt(#T1)*floor(block/sqrt(#T1)) + floor(element/sqrt(#T2))
			sumTensor[conjugate + 1][complex + 1][ne1] = sumT1[block + 1][element + 1][ne1]
		end
	end
end

--calculate RIXS conductivity tensor elements and collect them in a table with first index (conjugate) and second index (complex) both for xx,xy,xz,yx,yy,yz,zx,zy,zz

RIXS_Tensor = {}                                        
for conjugate = 1, #T1 do
	RIXS_Tensor[conjugate] = {}
	for complex = 1, #T2 do
		RIXS_Tensor[conjugate][complex] = {}
		for ne1 = 1, NE1 + 1 do
			 RIXS_Tensor[conjugate][complex][ne1] = Spectra.Sum(RIXSSpectra, sumTensor[conjugate][complex][ne1])
		end
	end
end

TimeEnd("Calculation RIXS Tensor")

------------------------------------------------------------------------------------
--now, we can print the tensor into 81 files in the data folder and work with them--
------------------------------------------------------------------------------------

TimeStart("Print RIXS Tensor")

--as we collect a lot of data, it makes sense to save the files in a separate folder

os.execute("mkdir data")

--print the tensor

for conjugate = 1, #T1 do
	for complex = 1, #T2 do
		for ne1 = 1, NE1 + 1 do
			 RIXS_Tensor[conjugate][complex][ne1].Print({{"file", string.format("data/RIXS_Tensor_Oh_%d_%d_%d.dat", conjugate, complex, ne1)}})
		end
	end
end

--it is also possible to rotate the polarization combinations to a basis of irreducible representations of Oh 
--(x, y, z)x(x, y, z) to (a1g, t1u, t1u, t1u, eg, eg, t2g, t2g, t2g)

f = math.sqrt(1/6)

rotMatU = { {d	,0	,0	,0	,d	,0	,0	,0	,d	},
			{0	,0	,0	,0	,0	,c	,0 	,-c	,0	},
			{0	,0	,c	,0	,0	,0	,-c	,0	,0	},
			{0	,c	,0	,-c	,0	,0	,0	,0	,0	},
			{-f	,0	,0	,0	,-f	,0	,0	,0	,2*f},
			{c	,0	,0	,0	,-c	,0	,0	,0	,0	},
			{0	,0	,c	,0	,0	,0	,c	,0	,0	},
			{0	,0	,0	,0	,0	,c	,0	,c	,0	},
			{0	,c	,0	,c	,0	,0	,0	,0	,0	}	
}

--the rotation is done by simple matrix multiplications: new[i][j] = U[i][n]*old[n][m]*U^dagger[m][j]

RIXS_Tensor_irreps = {}
for i = 1, 9 do
  RIXS_Tensor_irreps[i] = {}
  for j = 1, 9 do
RIXS_Tensor_irreps[i][j] = {}
  end
end    
	
for i = 1, 9 do
  for j = 1, 9 do
for ne1 = 1, NE1 + 1 do
  RIXS_Tensor_irreps[i][j][ne1] = RIXS_Tensor[1][1][1] - RIXS_Tensor[1][1][1]
  for m = 1, 9 do
	for n = 1, 9 do
	RIXS_Tensor_irreps[i][j][ne1] = RIXS_Tensor_irreps[i][j][ne1] + rotMatU[i][n]*RIXS_Tensor[n][m][ne1]*rotMatU[j][m]
	end
  end
end
  end
end

--we also print this tensor

for conjugate = 1, #T1 do
	for complex = 1, #T2 do
		for ne1 = 1, NE1 + 1 do
			 RIXS_Tensor_irreps[conjugate][complex][ne1].Print({{"file", string.format("data/RIXS_Tensor_Oh_irreps_%d_%d_%d.dat", conjugate, complex, ne1)}})
		end
	end
end

TimeEnd("Print RIXS Tensor")

TimePrint()
